<?php
// api.php - API de licencias para SendPlus
// Dominio: https://licencias.colegiopatriarcasj.com
// Compatibilidad: PHP >= 7.4
// Respuestas JSON, tokens de 30 días.

header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once("db.php");

function json_response($data, $code = 200) {
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function read_json_body() {
    $raw = file_get_contents("php://input");
    $data = json_decode($raw, true);
    if (!is_array($data)) { $data = []; }
    return $data;
}

function hash_password($pass) {
    // SHA-256 para compatibilidad; usa SIEMPRE HTTPS
    return hash('sha256', $pass);
}

function make_token($email) {
    return hash('sha256', $email . microtime(true) . random_bytes(16));
}

function get_bearer_token() {
    $headers = [];
    if (function_exists('getallheaders')) {
        $headers = getallheaders();
    } else {
        foreach ($_SERVER as $name => $value) {
            if (substr($name, 0, 5) == 'HTTP_') {
                $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                $headers[$key] = $value;
            }
        }
    }
    $auth = $headers['Authorization'] ?? $headers['authorization'] ?? '';
    if (stripos($auth, 'Bearer ') === 0) {
        return trim(substr($auth, 7));
    }
    return null;
}

$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$path = rtrim($path, '/');
$method = $_SERVER['REQUEST_METHOD'];

// --- Endpoint: /api/v1/auth/login ---
if ($method === 'POST' && (substr($path, -5) === '/login')) {
    $in = read_json_body();
    $email = trim($in['email'] ?? '');
    $password = $in['password'] ?? '';
    $machine_id = $in['machine_id'] ?? '';
    $app_id = $in['app_id'] ?? '';
    $version = $in['version'] ?? '';

    if ($email === '' || $password === '') {
        json_response(['error' => 'Datos incompletos'], 400);
    }

    try {
        $db = get_db();
        $stmt = $db->prepare('SELECT * FROM licencias WHERE email=? AND password_hash=? AND revoked=0');
        $stmt->execute([$email, hash('sha256', $password)]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$user) {
            json_response(['error' => 'Credenciales inválidas'], 401);
        }

        // Genera token y guarda device
        $token = make_token($email);
        $expires = time() + 60*60*24*30;

        $upd = $db->prepare('UPDATE licencias SET token=?, expires_at=?, machine_id=? WHERE email=?');
        $upd->execute([$token, $expires, $machine_id, $email]);

        json_response([
            'token' => $token,
            'expires_at' => $expires,
            'plan' => $user['plan'] ?? 'basic',
            'features' => new stdClass()
        ], 200);
    } catch (Exception $e) {
        json_response(['error' => 'Error del servidor'], 500);
    }
}

// --- Endpoint: /api/v1/license/verify ---
if ($method === 'POST' && (substr($path, -7) === '/verify')) {
    $token = get_bearer_token();
    if (!$token) {
        json_response(['valid' => false, 'message' => 'Falta token Bearer'], 401);
    }
    try {
        $db = get_db();
        $stmt = $db->prepare('SELECT * FROM licencias WHERE token=? AND revoked=0');
        $stmt->execute([$token]);
        $lic = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$lic) {
            json_response(['valid' => false, 'message' => 'Token inválido. Comuníquese con SendPlus al +50689319138.'], 401);
        }
        $now = time();
        $valid = ($lic['expires_at'] > $now);
        if (!$valid) {
            json_response(['valid' => false, 'message' => 'Licencia vencida. Comuníquese con SendPlus al +50689319138.'], 401);
        }
        json_response([
            'valid' => true,
            'expires_at' => intval($lic['expires_at']),
            'min_version' => '1.0.0',
            'revoked' => false
        ], 200);
    } catch (Exception $e) {
        json_response(['valid' => false, 'message' => 'Error del servidor'], 500);
    }
}

json_response(['error' => 'Ruta no encontrada'], 404);
?>